<?php
session_start();

// Verificar autenticación
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: login.php');
    exit;
}

require_once '../api/config.php';

// Obtener estadísticas del dashboard
function getDashboardStats() {
    $pdo = getDbConnection();
    
    $stats = [];
    
    // Total de licencias
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM licenses");
    $stats['total_licenses'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Licencias activas
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM licenses WHERE status = 'Active'");
    $stats['active_licenses'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Licencias expiradas
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM licenses WHERE status = 'Expired' OR expires_at < NOW()");
    $stats['expired_licenses'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Total de clientes
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM clients WHERE is_active = 1");
    $stats['total_clients'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Verificaciones hoy
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM license_logs WHERE DATE(created_at) = CURDATE()");
    $stats['today_verifications'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Licencias que expiran en 30 días
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM licenses WHERE status = 'Active' AND expires_at BETWEEN NOW() AND DATE_ADD(NOW(), INTERVAL 30 DAY)");
    $stats['expiring_soon'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    return $stats;
}

$stats = getDashboardStats();
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Sistema de Licencias GD</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../assets/css/admin.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-tachometer-alt"></i> Dashboard</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <div class="btn-group me-2">
                            <span class="badge bg-primary">
                                <i class="fas fa-clock"></i> 
                                <?= date('d/m/Y H:i:s') ?>
                            </span>
                        </div>
                    </div>
                </div>

                <!-- Tarjetas de estadísticas -->
                <div class="row">
                    <div class="col-xl-2 col-md-4 mb-4">
                        <div class="card border-left-primary shadow h-100 py-2">
                            <div class="card-body">
                                <div class="row no-gutters align-items-center">
                                    <div class="col mr-2">
                                        <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">
                                            Total Licencias
                                        </div>
                                        <div class="h5 mb-0 font-weight-bold text-gray-800">
                                            <?= number_format($stats['total_licenses']) ?>
                                        </div>
                                    </div>
                                    <div class="col-auto">
                                        <i class="fas fa-key fa-2x text-gray-300"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="col-xl-2 col-md-4 mb-4">
                        <div class="card border-left-success shadow h-100 py-2">
                            <div class="card-body">
                                <div class="row no-gutters align-items-center">
                                    <div class="col mr-2">
                                        <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                            Activas
                                        </div>
                                        <div class="h5 mb-0 font-weight-bold text-gray-800">
                                            <?= number_format($stats['active_licenses']) ?>
                                        </div>
                                    </div>
                                    <div class="col-auto">
                                        <i class="fas fa-check-circle fa-2x text-gray-300"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="col-xl-2 col-md-4 mb-4">
                        <div class="card border-left-danger shadow h-100 py-2">
                            <div class="card-body">
                                <div class="row no-gutters align-items-center">
                                    <div class="col mr-2">
                                        <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">
                                            Expiradas
                                        </div>
                                        <div class="h5 mb-0 font-weight-bold text-gray-800">
                                            <?= number_format($stats['expired_licenses']) ?>
                                        </div>
                                    </div>
                                    <div class="col-auto">
                                        <i class="fas fa-times-circle fa-2x text-gray-300"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="col-xl-2 col-md-4 mb-4">
                        <div class="card border-left-info shadow h-100 py-2">
                            <div class="card-body">
                                <div class="row no-gutters align-items-center">
                                    <div class="col mr-2">
                                        <div class="text-xs font-weight-bold text-info text-uppercase mb-1">
                                            Clientes
                                        </div>
                                        <div class="h5 mb-0 font-weight-bold text-gray-800">
                                            <?= number_format($stats['total_clients']) ?>
                                        </div>
                                    </div>
                                    <div class="col-auto">
                                        <i class="fas fa-users fa-2x text-gray-300"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="col-xl-2 col-md-4 mb-4">
                        <div class="card border-left-warning shadow h-100 py-2">
                            <div class="card-body">
                                <div class="row no-gutters align-items-center">
                                    <div class="col mr-2">
                                        <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">
                                            Verificaciones Hoy
                                        </div>
                                        <div class="h5 mb-0 font-weight-bold text-gray-800">
                                            <?= number_format($stats['today_verifications']) ?>
                                        </div>
                                    </div>
                                    <div class="col-auto">
                                        <i class="fas fa-chart-line fa-2x text-gray-300"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="col-xl-2 col-md-4 mb-4">
                        <div class="card border-left-warning shadow h-100 py-2">
                            <div class="card-body">
                                <div class="row no-gutters align-items-center">
                                    <div class="col mr-2">
                                        <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">
                                            Expiran Pronto
                                        </div>
                                        <div class="h5 mb-0 font-weight-bold text-gray-800">
                                            <?= number_format($stats['expiring_soon']) ?>
                                        </div>
                                    </div>
                                    <div class="col-auto">
                                        <i class="fas fa-exclamation-triangle fa-2x text-gray-300"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Acciones rápidas -->
                <div class="row">
                    <div class="col-lg-6">
                        <div class="card shadow mb-4">
                            <div class="card-header py-3">
                                <h6 class="m-0 font-weight-bold text-primary">
                                    <i class="fas fa-bolt"></i> Acciones Rápidas
                                </h6>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-6 mb-3">
                                        <a href="clients.php?action=add" class="btn btn-success btn-block">
                                            <i class="fas fa-user-plus"></i> Nuevo Cliente
                                        </a>
                                    </div>
                                    <div class="col-6 mb-3">
                                        <a href="licenses.php?action=add" class="btn btn-primary btn-block">
                                            <i class="fas fa-key"></i> Nueva Licencia
                                        </a>
                                    </div>
                                    <div class="col-6 mb-3">
                                        <a href="logs.php" class="btn btn-info btn-block">
                                            <i class="fas fa-list-alt"></i> Ver Logs
                                        </a>
                                    </div>
                                    <div class="col-6 mb-3">
                                        <a href="settings.php" class="btn btn-secondary btn-block">
                                            <i class="fas fa-cog"></i> Configuración
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="col-lg-6">
                        <div class="card shadow mb-4">
                            <div class="card-header py-3">
                                <h6 class="m-0 font-weight-bold text-primary">
                                    <i class="fas fa-info-circle"></i> Estado del Sistema
                                </h6>
                            </div>
                            <div class="card-body">
                                <div class="mb-3">
                                    <strong>API Status:</strong>
                                    <span class="badge bg-success">
                                        <i class="fas fa-check"></i> Online
                                    </span>
                                </div>
                                <div class="mb-3">
                                    <strong>Base de Datos:</strong>
                                    <span class="badge bg-success">
                                        <i class="fas fa-check"></i> Conectada
                                    </span>
                                </div>
                                <div class="mb-3">
                                    <strong>Endpoint API:</strong>
                                    <code><?= API_BASE_URL ?>/verify.php</code>
                                </div>
                                <div class="mb-3">
                                    <strong>Versión:</strong>
                                    <span class="badge bg-info"><?= API_VERSION ?></span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Licencias que expiran pronto -->
                <?php if ($stats['expiring_soon'] > 0): ?>
                <div class="card shadow mb-4">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-warning">
                            <i class="fas fa-exclamation-triangle"></i> 
                            Licencias que Expiran en los Próximos 30 Días
                        </h6>
                    </div>
                    <div class="card-body">
                        <?php
                        $pdo = getDbConnection();
                        $stmt = $pdo->query("
                            SELECT l.license_key, l.expires_at, c.name as client_name, c.email 
                            FROM licenses l 
                            JOIN clients c ON l.client_id = c.id 
                            WHERE l.status = 'Active' 
                            AND l.expires_at BETWEEN NOW() AND DATE_ADD(NOW(), INTERVAL 30 DAY)
                            ORDER BY l.expires_at ASC
                            LIMIT 10
                        ");
                        $expiring = $stmt->fetchAll(PDO::FETCH_ASSOC);
                        ?>
                        
                        <div class="table-responsive">
                            <table class="table table-bordered" width="100%" cellspacing="0">
                                <thead>
                                    <tr>
                                        <th>Licencia</th>
                                        <th>Cliente</th>
                                        <th>Email</th>
                                        <th>Expira</th>
                                        <th>Días Restantes</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($expiring as $license): ?>
                                    <tr>
                                        <td><code><?= htmlspecialchars($license['license_key']) ?></code></td>
                                        <td><?= htmlspecialchars($license['client_name']) ?></td>
                                        <td><?= htmlspecialchars($license['email']) ?></td>
                                        <td><?= date('d/m/Y', strtotime($license['expires_at'])) ?></td>
                                        <td>
                                            <?php 
                                            $days = ceil((strtotime($license['expires_at']) - time()) / 86400);
                                            $class = $days <= 7 ? 'danger' : ($days <= 15 ? 'warning' : 'info');
                                            ?>
                                            <span class="badge bg-<?= $class ?>"><?= $days ?> días</span>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                <?php endif; ?>

            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="../assets/js/admin.js"></script>
</body>
</html> 