<?php
/**
 * Funciones Auxiliares para la API del Sistema de Licencias
 * Grupos Digitales - License System
 */

require_once __DIR__ . '/config.php';

/**
 * Función para registrar logs de la API
 */
function logApiRequest($level, $message, $context = []) {
    if (!ENABLE_API_LOGGING) return;
    
    $logLevels = ['DEBUG' => 0, 'INFO' => 1, 'WARNING' => 2, 'ERROR' => 3];
    $currentLevel = $logLevels[LOG_LEVEL] ?? 1;
    $messageLevel = $logLevels[$level] ?? 1;
    
    if ($messageLevel < $currentLevel) return;
    
    $timestamp = date('Y-m-d H:i:s');
    $contextStr = !empty($context) ? ' | ' . json_encode($context) : '';
    $logEntry = "[{$timestamp}] [{$level}] {$message}{$contextStr}" . PHP_EOL;
    
    @file_put_contents(LOG_FILE, $logEntry, FILE_APPEND | LOCK_EX);
}

/**
 * Función para enviar respuesta de la API
 */
function sendApiResponse($data, $statusCode = 200, $format = null) {
    $format = $format ?: ($_GET['format'] ?? DEFAULT_RESPONSE_FORMAT);
    
    // Configurar headers
    foreach (DEFAULT_HEADERS as $header => $value) {
        header("{$header}: {$value}");
    }
    
    if (ENABLE_CORS) {
        header('Access-Control-Allow-Origin: ' . ALLOWED_ORIGINS);
        header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
        header('Access-Control-Allow-Headers: Content-Type, Authorization');
    }
    
    http_response_code($statusCode);
    
    // Formatear respuesta según el formato solicitado
    switch (strtolower($format)) {
        case 'xml':
            header('Content-Type: application/xml; charset=UTF-8');
            echo formatXmlResponse($data);
            break;
            
        case 'pipes':
            header('Content-Type: text/plain; charset=UTF-8');
            echo formatPipesResponse($data);
            break;
            
        case 'json':
        default:
            header('Content-Type: application/json; charset=UTF-8');
            echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
            break;
    }
    
    exit;
}

/**
 * Función para formatear respuesta en XML
 */
function formatXmlResponse($data) {
    $xml = new SimpleXMLElement('<license/>');
    
    foreach ($data as $key => $value) {
        if (is_array($value)) {
            $value = json_encode($value);
        }
        $xml->addChild($key, htmlspecialchars($value));
    }
    
    $dom = new DOMDocument('1.0', 'UTF-8');
    $dom->formatOutput = true;
    $dom->loadXML($xml->asXML());
    
    return $dom->saveXML();
}

/**
 * Función para formatear respuesta en formato pipes (compatibilidad WHMCS)
 */
function formatPipesResponse($data) {
    $status = $data['status'] ?? 'Invalid';
    $extra = [];
    
    foreach ($data as $key => $value) {
        if ($key !== 'status' && $key !== 'localkey') {
            if (is_array($value)) {
                $value = implode(',', $value);
            }
            $extra[] = "{$key}=" . urlencode($value);
        }
    }
    
    $extraStr = implode(';', $extra);
    $localkey = $data['localkey'] ?? '';
    
    return "{$status}|{$extraStr}|{$localkey}";
}

/**
 * Función para enviar respuesta de error
 */
function sendErrorResponse($errorCode, $statusCode = 400, $details = null) {
    $message = ERROR_MESSAGES[$errorCode] ?? 'Error desconocido';
    
    $response = [
        'status' => 'Error',
        'error_code' => $errorCode,
        'message' => $message
    ];
    
    if ($details) {
        $response['details'] = $details;
    }
    
    logApiRequest('ERROR', "Error: {$errorCode} - {$message}", [
        'ip' => getClientIp(),
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
        'details' => $details
    ]);
    
    sendApiResponse($response, $statusCode);
}

/**
 * Función para obtener la IP del cliente
 */
function getClientIp() {
    $headers = [
        'HTTP_CF_CONNECTING_IP',     // Cloudflare
        'HTTP_CLIENT_IP',            // Proxy
        'HTTP_X_FORWARDED_FOR',      // Load balancer/proxy
        'HTTP_X_FORWARDED',          // Proxy
        'HTTP_X_CLUSTER_CLIENT_IP',  // Cluster
        'HTTP_FORWARDED_FOR',        // Proxy
        'HTTP_FORWARDED',            // Proxy
        'REMOTE_ADDR'                // Standard
    ];
    
    foreach ($headers as $header) {
        if (!empty($_SERVER[$header])) {
            $ips = explode(',', $_SERVER[$header]);
            $ip = trim($ips[0]);
            
            if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                return $ip;
            }
        }
    }
    
    return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
}

/**
 * Función para verificar rate limiting
 */
function checkRateLimit($ip) {
    $cacheFile = CACHE_DIR . 'rate_limit_' . md5($ip);
    $now = time();
    $hourAgo = $now - 3600;
    
    // Leer requests existentes
    $requests = [];
    if (file_exists($cacheFile)) {
        $data = @file_get_contents($cacheFile);
        if ($data) {
            $requests = json_decode($data, true) ?: [];
        }
    }
    
    // Filtrar requests de la última hora
    $requests = array_filter($requests, function($timestamp) use ($hourAgo) {
        return $timestamp > $hourAgo;
    });
    
    // Verificar límite
    if (count($requests) >= RATE_LIMIT_REQUESTS) {
        return false;
    }
    
    // Agregar nueva request
    $requests[] = $now;
    
    // Guardar en cache
    @file_put_contents($cacheFile, json_encode($requests), LOCK_EX);
    
    return true;
}

/**
 * Función para verificar si una licencia existe y obtener sus datos
 */
function getLicenseData($licenseKey) {
    try {
        $pdo = getDatabaseConnection();
        
        $stmt = $pdo->prepare("
            SELECT l.*, c.name as client_name, c.email as client_email, c.company
            FROM licenses l
            JOIN clients c ON l.client_id = c.id
            WHERE l.license_key = ? AND c.is_active = 1
        ");
        
        $stmt->execute([$licenseKey]);
        $license = $stmt->fetch();
        
        if (!$license) {
            return null;
        }
        
        // Decodificar dominios permitidos
        $license['domains_array'] = $license['domains'] ? json_decode($license['domains'], true) : [];
        
        return $license;
    } catch (Exception $e) {
        logApiRequest('ERROR', 'Database error getting license data', ['error' => $e->getMessage()]);
        return false;
    }
}

/**
 * Función para verificar si un dominio está autorizado para una licencia
 */
function isDomainAuthorized($license, $domain) {
    if (empty($license['domains_array'])) {
        return true; // Sin restricciones de dominio
    }
    
    $domain = strtolower(trim($domain));
    
    foreach ($license['domains_array'] as $allowedDomain) {
        $allowedDomain = strtolower(trim($allowedDomain));
        
        // Coincidencia exacta
        if ($domain === $allowedDomain) {
            return true;
        }
        
        // Wildcard subdominios
        if (strpos($allowedDomain, '*.') === 0) {
            $pattern = str_replace('*.', '', $allowedDomain);
            if (strpos($domain, $pattern) !== false) {
                return true;
            }
        }
    }
    
    return false;
}

/**
 * Función para verificar el estado de una licencia
 */
function checkLicenseStatus($license) {
    // Verificar expiración
    if ($license['expires_at'] && strtotime($license['expires_at']) < time()) {
        return [
            'status' => 'Expired',
            'message' => 'La licencia ha expirado el ' . date('d/m/Y', strtotime($license['expires_at']))
        ];
    }
    
    // Verificar estado
    switch ($license['status']) {
        case 'Active':
            return [
                'status' => 'Active',
                'message' => 'Licencia activa y válida'
            ];
            
        case 'Suspended':
            return [
                'status' => 'Suspended',
                'message' => 'La licencia está suspendida'
            ];
            
        case 'Cancelled':
            return [
                'status' => 'Cancelled',
                'message' => 'La licencia ha sido cancelada'
            ];
            
        default:
            return [
                'status' => 'Invalid',
                'message' => 'Estado de licencia desconocido'
            ];
    }
}

/**
 * Función para crear una clave local
 */
function createLocalKey($license, $domain) {
    $localKeyData = [
        'license_key' => $license['license_key'],
        'status' => $license['status'],
        'client_name' => $license['client_name'],
        'product_name' => $license['product_name'],
        'domain' => $domain,
        'created_at' => date('Y-m-d H:i:s'),
        'expires_at' => date('Y-m-d H:i:s', strtotime('+' . LOCAL_KEY_VALIDITY_DAYS . ' days')),
        'checksum' => md5($license['license_key'] . $domain . API_SECRET_KEY)
    ];
    
    $encoded = base64_encode(json_encode($localKeyData));
    $signature = hash_hmac('sha256', $encoded, API_SECRET_KEY);
    
    $localKey = $encoded . '.' . $signature;
    
    // Guardar en base de datos para futuras validaciones
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare("
            INSERT INTO local_keys (license_key, domain, local_key_data, expires_at)
            VALUES (?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE 
                local_key_data = VALUES(local_key_data),
                expires_at = VALUES(expires_at),
                created_at = NOW()
        ");
        
        $stmt->execute([
            $license['license_key'],
            $domain,
            $localKey,
            date('Y-m-d H:i:s', strtotime('+' . LOCAL_KEY_VALIDITY_DAYS . ' days'))
        ]);
    } catch (Exception $e) {
        logApiRequest('WARNING', 'Failed to save local key to database', ['error' => $e->getMessage()]);
    }
    
    return $localKey;
}

/**
 * Función para registrar el log de verificación
 */
function logLicenseVerification($licenseKey, $domain, $ip, $status, $response, $errorMessage = null) {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare("
            INSERT INTO license_logs (license_key, domain, ip_address, user_agent, status, response, error_message)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $licenseKey,
            $domain,
            $ip,
            $_SERVER['HTTP_USER_AGENT'] ?? '',
            $status,
            json_encode($response),
            $errorMessage
        ]);
    } catch (Exception $e) {
        logApiRequest('ERROR', 'Failed to log verification', ['error' => $e->getMessage()]);
    }
}

/**
 * Función para validar el formato de la clave de licencia
 */
function validateLicenseFormat($licenseKey) {
    return preg_match(LICENSE_PATTERN, $licenseKey);
}

/**
 * Función para limpiar datos de entrada
 */
function sanitizeInput($input) {
    return trim(strip_tags($input));
}

/**
 * Función para generar una nueva clave de licencia
 */
function generateLicenseKey() {
    $characters = 'abcdefghijklmnopqrstuvwxyz0123456789';
    $key = LICENSE_PREFIX;
    
    for ($i = 0; $i < LICENSE_LENGTH; $i++) {
        $key .= $characters[random_int(0, strlen($characters) - 1)];
    }
    
    return $key;
}

/**
 * Función para verificar la integridad de la solicitud
 */
function validateRequest() {
    // Verificar método de solicitud
    if ($_SERVER['REQUEST_METHOD'] !== 'POST' && $_SERVER['REQUEST_METHOD'] !== 'GET') {
        sendErrorResponse('INVALID_REQUEST', 405);
    }
    
    // Verificar modo mantenimiento
    if (isMaintenanceMode()) {
        sendErrorResponse('MAINTENANCE_MODE', 503);
    }
    
    // Verificar rate limiting
    $ip = getClientIp();
    if (!checkRateLimit($ip)) {
        sendErrorResponse('RATE_LIMIT_EXCEEDED', 429);
    }
    
    return true;
} 