/**
 * JavaScript para el Panel de Administración
 * Sistema de Licencias - Grupos Digitales
 */

// Inicialización cuando el DOM está listo
document.addEventListener('DOMContentLoaded', function() {
    initializeAdmin();
});

/**
 * Inicializar funcionalidades del panel admin
 */
function initializeAdmin() {
    // Auto-cerrar alertas después de 5 segundos
    autoCloseAlerts();
    
    // Tooltips para botones
    initializeTooltips();
    
    // Confirmaciones de eliminación
    initializeDeleteConfirmations();
    
    // Actualizar timestamps en tiempo real
    updateTimestamps();
    
    // Validaciones de formularios
    initializeFormValidations();
    
    // Búsqueda en tablas
    initializeTableSearch();
    
    // Copiar al portapapeles
    initializeCopyToClipboard();
    
    // Actualizar estadísticas automáticamente
    if (window.location.pathname.includes('index.php') || window.location.pathname.endsWith('/admin/')) {
        autoUpdateStats();
    }
}

/**
 * Auto-cerrar alertas
 */
function autoCloseAlerts() {
    const alerts = document.querySelectorAll('.alert:not(.alert-persistent)');
    alerts.forEach(alert => {
        setTimeout(() => {
            const bsAlert = new bootstrap.Alert(alert);
            bsAlert.close();
        }, 5000);
    });
}

/**
 * Inicializar tooltips
 */
function initializeTooltips() {
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"], [title]'));
    tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
}

/**
 * Confirmaciones de eliminación mejoradas
 */
function initializeDeleteConfirmations() {
    const deleteLinks = document.querySelectorAll('a[href*="delete"], .btn-delete');
    deleteLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            
            const itemType = this.dataset.itemType || 'elemento';
            const itemName = this.dataset.itemName || '';
            const confirmText = `¿Estás seguro de eliminar ${itemType}${itemName ? ' "' + itemName + '"' : ''}?`;
            
            showConfirmModal(
                'Confirmar Eliminación',
                confirmText,
                'Esta acción no se puede deshacer.',
                'Eliminar',
                'btn-danger',
                () => {
                    window.location.href = this.href;
                }
            );
        });
    });
}

/**
 * Actualizar timestamps relativos
 */
function updateTimestamps() {
    const timestamps = document.querySelectorAll('[data-timestamp]');
    timestamps.forEach(element => {
        const timestamp = parseInt(element.dataset.timestamp) * 1000;
        element.textContent = getRelativeTime(timestamp);
    });
    
    // Actualizar cada minuto
    setInterval(() => {
        timestamps.forEach(element => {
            const timestamp = parseInt(element.dataset.timestamp) * 1000;
            element.textContent = getRelativeTime(timestamp);
        });
    }, 60000);
}

/**
 * Obtener tiempo relativo
 */
function getRelativeTime(timestamp) {
    const now = Date.now();
    const diff = now - timestamp;
    const seconds = Math.floor(diff / 1000);
    const minutes = Math.floor(seconds / 60);
    const hours = Math.floor(minutes / 60);
    const days = Math.floor(hours / 24);
    
    if (days > 0) return `hace ${days} día${days > 1 ? 's' : ''}`;
    if (hours > 0) return `hace ${hours} hora${hours > 1 ? 's' : ''}`;
    if (minutes > 0) return `hace ${minutes} minuto${minutes > 1 ? 's' : ''}`;
    return 'hace unos segundos';
}

/**
 * Validaciones de formularios
 */
function initializeFormValidations() {
    const forms = document.querySelectorAll('form[data-validate]');
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            if (!validateForm(this)) {
                e.preventDefault();
                e.stopPropagation();
            }
            this.classList.add('was-validated');
        });
    });
}

/**
 * Validar formulario
 */
function validateForm(form) {
    let isValid = true;
    
    // Validar campos requeridos
    const requiredFields = form.querySelectorAll('[required]');
    requiredFields.forEach(field => {
        if (!field.value.trim()) {
            showFieldError(field, 'Este campo es requerido');
            isValid = false;
        } else {
            clearFieldError(field);
        }
    });
    
    // Validar emails
    const emailFields = form.querySelectorAll('input[type="email"]');
    emailFields.forEach(field => {
        if (field.value && !isValidEmail(field.value)) {
            showFieldError(field, 'Email inválido');
            isValid = false;
        }
    });
    
    return isValid;
}

/**
 * Mostrar error en campo
 */
function showFieldError(field, message) {
    clearFieldError(field);
    field.classList.add('is-invalid');
    
    const feedback = document.createElement('div');
    feedback.className = 'invalid-feedback';
    feedback.textContent = message;
    field.parentNode.appendChild(feedback);
}

/**
 * Limpiar error de campo
 */
function clearFieldError(field) {
    field.classList.remove('is-invalid');
    const feedback = field.parentNode.querySelector('.invalid-feedback');
    if (feedback) {
        feedback.remove();
    }
}

/**
 * Validar email
 */
function isValidEmail(email) {
    const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return re.test(email);
}

/**
 * Búsqueda en tablas
 */
function initializeTableSearch() {
    const searchInputs = document.querySelectorAll('[data-table-search]');
    searchInputs.forEach(input => {
        const targetTable = document.querySelector(input.dataset.tableSearch);
        if (targetTable) {
            input.addEventListener('input', function() {
                filterTable(targetTable, this.value);
            });
        }
    });
}

/**
 * Filtrar tabla
 */
function filterTable(table, searchTerm) {
    const rows = table.querySelectorAll('tbody tr');
    const term = searchTerm.toLowerCase();
    
    rows.forEach(row => {
        const text = row.textContent.toLowerCase();
        row.style.display = text.includes(term) ? '' : 'none';
    });
}

/**
 * Copiar al portapapeles
 */
function initializeCopyToClipboard() {
    const copyButtons = document.querySelectorAll('[data-copy]');
    copyButtons.forEach(button => {
        button.addEventListener('click', function() {
            const text = this.dataset.copy || this.textContent;
            copyToClipboard(text);
            
            // Feedback visual
            const originalText = this.innerHTML;
            this.innerHTML = '<i class="fas fa-check"></i> Copiado!';
            setTimeout(() => {
                this.innerHTML = originalText;
            }, 2000);
        });
    });
}

/**
 * Copiar texto al portapapeles
 */
async function copyToClipboard(text) {
    try {
        await navigator.clipboard.writeText(text);
    } catch (err) {
        // Fallback para navegadores antiguos
        const textArea = document.createElement('textarea');
        textArea.value = text;
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
    }
}

/**
 * Auto-actualizar estadísticas del dashboard
 */
function autoUpdateStats() {
    setInterval(async () => {
        try {
            const response = await fetch('api/stats.php');
            const stats = await response.json();
            updateDashboardStats(stats);
        } catch (error) {
            console.log('Error actualizando estadísticas:', error);
        }
    }, 30000); // Cada 30 segundos
}

/**
 * Actualizar estadísticas en el dashboard
 */
function updateDashboardStats(stats) {
    Object.entries(stats).forEach(([key, value]) => {
        const element = document.querySelector(`[data-stat="${key}"]`);
        if (element) {
            element.textContent = value.toLocaleString();
        }
    });
}

/**
 * Mostrar modal de confirmación
 */
function showConfirmModal(title, message, details, confirmText, confirmClass, onConfirm) {
    // Crear modal si no existe
    let modal = document.getElementById('confirmModal');
    if (!modal) {
        modal = createConfirmModal();
        document.body.appendChild(modal);
    }
    
    // Actualizar contenido
    modal.querySelector('.modal-title').textContent = title;
    modal.querySelector('.modal-body p').textContent = message;
    modal.querySelector('.modal-body small').textContent = details;
    
    const confirmBtn = modal.querySelector('.btn-confirm');
    confirmBtn.textContent = confirmText;
    confirmBtn.className = `btn ${confirmClass}`;
    
    // Manejar confirmación
    confirmBtn.onclick = () => {
        const bsModal = bootstrap.Modal.getInstance(modal);
        bsModal.hide();
        onConfirm();
    };
    
    // Mostrar modal
    const bsModal = new bootstrap.Modal(modal);
    bsModal.show();
}

/**
 * Crear modal de confirmación
 */
function createConfirmModal() {
    const modal = document.createElement('div');
    modal.className = 'modal fade';
    modal.id = 'confirmModal';
    modal.innerHTML = `
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p></p>
                    <small class="text-muted"></small>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="button" class="btn btn-confirm"></button>
                </div>
            </div>
        </div>
    `;
    return modal;
}

/**
 * Mostrar notificación toast
 */
function showToast(message, type = 'info') {
    const toastContainer = getOrCreateToastContainer();
    
    const toast = document.createElement('div');
    toast.className = `toast align-items-center text-white bg-${type} border-0`;
    toast.innerHTML = `
        <div class="d-flex">
            <div class="toast-body">
                ${message}
            </div>
            <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
        </div>
    `;
    
    toastContainer.appendChild(toast);
    
    const bsToast = new bootstrap.Toast(toast);
    bsToast.show();
    
    // Remover del DOM después de ocultarse
    toast.addEventListener('hidden.bs.toast', () => {
        toast.remove();
    });
}

/**
 * Obtener o crear contenedor de toasts
 */
function getOrCreateToastContainer() {
    let container = document.getElementById('toastContainer');
    if (!container) {
        container = document.createElement('div');
        container.id = 'toastContainer';
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        container.style.zIndex = '1055';
        document.body.appendChild(container);
    }
    return container;
}

/**
 * Formatear números
 */
function formatNumber(num) {
    return new Intl.NumberFormat('es-ES').format(num);
}

/**
 * Formatear fechas
 */
function formatDate(date, options = {}) {
    const defaultOptions = {
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit'
    };
    
    return new Intl.DateTimeFormat('es-ES', { ...defaultOptions, ...options }).format(new Date(date));
}

/**
 * Detectar tema del sistema
 */
function getSystemTheme() {
    return window.matchMedia('(prefers-color-scheme: dark)').matches ? 'dark' : 'light';
}

/**
 * Funciones de utilidad global
 */
window.AdminUtils = {
    showToast,
    showConfirmModal,
    copyToClipboard,
    formatNumber,
    formatDate,
    getSystemTheme
};

// Log de inicialización
console.log('Sistema de Licencias GD - Panel Admin cargado correctamente'); 